import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getEmployeesCollection } from "@/lib/database/collections"

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const { id } = params

    if (!ObjectId.isValid(id)) {
      return NextResponse.json(
        {
          success: false,
          error: "Invalid employee ID",
        },
        { status: 400 },
      )
    }

    console.log("[v0] Fetching employee:", id)
    const employeesCollection = await getEmployeesCollection()

    const employee = await employeesCollection.findOne({ _id: new ObjectId(id) })

    if (!employee) {
      return NextResponse.json(
        {
          success: false,
          error: "Employee not found",
        },
        { status: 404 },
      )
    }

    console.log("[v0] Employee found successfully")

    return NextResponse.json({
      success: true,
      data: employee,
    })
  } catch (error) {
    console.error("[v0] Error fetching employee:", error)
    return NextResponse.json(
      {
        success: false,
        error: "Failed to fetch employee",
      },
      { status: 500 },
    )
  }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const { id } = params

    if (!ObjectId.isValid(id)) {
      return NextResponse.json(
        {
          success: false,
          error: "Invalid employee ID",
        },
        { status: 400 },
      )
    }

    console.log("[v0] Updating employee:", id)
    const body = await request.json()

    const employeesCollection = await getEmployeesCollection()

    const updateData: any = {
      updatedAt: new Date(),
    }

    if (body.name) updateData.name = body.name
    if (body.email !== undefined) updateData.email = body.email
    if (body.phone !== undefined) updateData.phone = body.phone
    if (body.position) updateData.position = body.position
    if (body.branchId && ObjectId.isValid(body.branchId)) {
      updateData.branchId = new ObjectId(body.branchId)
    }
    if (body.hourlyRate !== undefined) updateData.hourlyRate = body.hourlyRate
    if (body.isActive !== undefined) updateData.isActive = body.isActive
    if (body.startDate) updateData.startDate = new Date(body.startDate)
    if (body.endDate) updateData.endDate = new Date(body.endDate)

    const result = await employeesCollection.updateOne({ _id: new ObjectId(id) }, { $set: updateData })

    if (result.matchedCount === 0) {
      return NextResponse.json(
        {
          success: false,
          error: "Employee not found",
        },
        { status: 404 },
      )
    }

    const updatedEmployee = await employeesCollection.findOne({ _id: new ObjectId(id) })

    console.log("[v0] Employee updated successfully")

    return NextResponse.json({
      success: true,
      data: updatedEmployee,
    })
  } catch (error) {
    console.error("[v0] Error updating employee:", error)
    return NextResponse.json(
      {
        success: false,
        error: "Failed to update employee",
      },
      { status: 500 },
    )
  }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const { id } = params

    if (!ObjectId.isValid(id)) {
      return NextResponse.json(
        {
          success: false,
          error: "Invalid employee ID",
        },
        { status: 400 },
      )
    }

    console.log("[v0] Deleting employee:", id)
    const employeesCollection = await getEmployeesCollection()

    const result = await employeesCollection.deleteOne({ _id: new ObjectId(id) })

    if (result.deletedCount === 0) {
      return NextResponse.json(
        {
          success: false,
          error: "Employee not found",
        },
        { status: 404 },
      )
    }

    console.log("[v0] Employee deleted successfully")

    return NextResponse.json({
      success: true,
      message: "Employee deleted successfully",
    })
  } catch (error) {
    console.error("[v0] Error deleting employee:", error)
    return NextResponse.json(
      {
        success: false,
        error: "Failed to delete employee",
      },
      { status: 500 },
    )
  }
}
